import json
import pandas as pd
from typing import List, Dict, Any, Optional

class Recorder:
    """
    A flexible trial recorder.
    
    Supports recording config, score, elapsed time, or any other metadata.
    Can save/load in JSON or CSV formats.
    """
    def __init__(self):
        self.records: List[Dict[str, Any]] = []

    def log(self, record: Dict[str, Any]) -> None:
        """
        Log a single trial record.

        Args:
            record: A dictionary containing any information (e.g., config, score, elapsed_time, etc.)
        """
        self.records.append(record)

    def save(self, filepath: str, format: str = "json") -> None:
        """
        Save all records to a file.

        Args:
            filepath: Output file path.
            format: "json" or "csv"
        """
        if format == "json":
            with open(filepath, "w") as f:
                json.dump(self.records, f, indent=4)
        elif format == "csv":
            df = pd.DataFrame(self.records)
            df.to_csv(filepath, index=False)
        else:
            raise ValueError(f"Unsupported format: {format}")

    def load(self, filepath: str, format: str = "json") -> None:
        """
        Load records from a file.

        Args:
            filepath: Input file path.
            format: "json" or "csv"
        """
        if format == "json":
            with open(filepath, "r") as f:
                self.records = json.load(f)
        elif format == "csv":
            df = pd.read_csv(filepath)
            self.records = df.to_dict(orient="records")
        else:
            raise ValueError(f"Unsupported format: {format}")

    def get_all(self) -> List[Dict[str, Any]]:
        return self.records

    def reset(self) -> None:
        self.records = []
